var VisualProfiler;
(function (VisualProfiler) {
    (function (Graph) {
        "use strict";
        (function (TickMarkType) {
            TickMarkType._map = [];
            TickMarkType.Big = 0;
            TickMarkType.Medium = 1;
            TickMarkType.Small = 2;
        })(Graph.TickMarkType || (Graph.TickMarkType = {}));
        var TickMarkType = Graph.TickMarkType;
        (function (MarkType) {
            MarkType._map = [];
            MarkType.LifeCycleEvent = 1;
            MarkType.UserMark = 2;
        })(Graph.MarkType || (Graph.MarkType = {}));
        var MarkType = Graph.MarkType;
        var RulerConfig = (function () {
            function RulerConfig(containerId) {
                this.containerId = "rulerContainer";
                this.className = "ruler-container";
                this.id = "";
                this.doubleSlider = {
                    containerId: "sliderContainer",
                    className: "ruler-body",
                    id: "rulerBody",
                    height: 1.8,
                    events: [
                        RulerEvents.Filter, 
                        RulerEvents.Zoom
                    ],
                    leftSlider: {
                        left: 0,
                        width: 5,
                        className: "ruler-slider"
                    },
                    rightSlider: {
                        left: 0,
                        width: 5,
                        className: "ruler-slider"
                    },
                    bar: {
                        className: "ruler-bar",
                        id: "rulerBar",
                        left: 40,
                        right: 40
                    },
                    timeRange: new VisualProfiler.TimeSpan(),
                    step: 1,
                    markSeries: [],
                    minimumRangeInPixel: 10,
                    minimumZoomLevelMsec: 50
                };
                this.header = {
                    containerId: "rulerHeaderContainer",
                    className: "ruler-header",
                    titleConfig: {
                        containerId: "rulerTitleContainer",
                        className: "ruler-title-container",
                        text: "Diagnostic Session",
                        isExpanded: true,
                        description: ""
                    },
                    legendConfig: {
                        containerId: "rulerLegendContainer",
                        className: "ruler-legend-container",
                        data: []
                    }
                };
                if(containerId) {
                    this.containerId = containerId;
                } else {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1051"));
                }
            }
            return RulerConfig;
        })();
        Graph.RulerConfig = RulerConfig;        
        var RulerEvents = (function () {
            function RulerEvents() { }
            Object.defineProperty(RulerEvents, "Filter", {
                get: function () {
                    return "filterTimeRange";
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(RulerEvents, "Zoom", {
                get: function () {
                    return "zoomTimeRange";
                },
                enumerable: true,
                configurable: true
            });
            return RulerEvents;
        })();
        Graph.RulerEvents = RulerEvents;        
        var Ruler = (function () {
            function Ruler(config) {
                this._rulerAreaCss = "ruler-area";
                if(config && config.containerId) {
                    this._container = document.getElementById(config.containerId);
                    if(this._container) {
                        this._config = config;
                        this.initialize(this._config);
                    } else {
                        throw new Error(Plugin.Resources.getErrorString("JSPerf.1051"));
                    }
                } else {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1050"));
                }
            }
            Ruler.prototype.deinitialize = function () {
                if(this._doubleSlider) {
                    this._doubleSlider.deinitialize();
                }
            };
            Ruler.prototype.initialize = function (object) {
                for(var property in object) {
                    if(object[property] && (property === "id" || property === "containerId")) {
                        object[property] += RulerUtilities.getRandomNumber();
                    } else {
                        if(typeof object[property] === "object" && property !== "data" && property !== "markSeries") {
                            this.initialize(object[property]);
                        }
                    }
                }
            };
            Ruler.prototype.setTimeRange = function (timespan) {
                this._doubleSlider.setTimeRange(timespan);
            };
            Ruler.prototype.zoomIn = function () {
                this._doubleSlider.onZoomIn();
            };
            Ruler.prototype.resetZoom = function () {
                this._doubleSlider.onResetZoom();
            };
            Ruler.prototype.clearSelection = function () {
                this._doubleSlider.clearSelection();
            };
            Ruler.prototype.addEventListener = function (eventType, func) {
                this._doubleSlider.addEventListener(eventType, func);
            };
            Ruler.prototype.removeEventListener = function (eventType, func) {
                this._doubleSlider.removeEventListener(eventType, func);
            };
            Ruler.prototype.setHandlePosition = function (timeRange) {
                this._doubleSlider.SliderHandlePosition = timeRange;
            };
            Object.defineProperty(Ruler.prototype, "Series", {
                get: function () {
                    return this._config.doubleSlider.markSeries;
                },
                enumerable: true,
                configurable: true
            });
            Ruler.prototype.render = function () {
                while(this._container.hasChildNodes()) {
                    this._container.removeChild(this._container.firstChild);
                }
                this._container.className = this._config.className;
                this._header = document.createElement("div");
                this._header.className = this._config.header.className;
                this._header.id = this._config.header.containerId;
                this._container.appendChild(this._header);
                this._body = document.createElement("div");
                this._body.id = this._config.doubleSlider.containerId;
                this._body.style.height = this._config.doubleSlider.height + "em";
                this._container.appendChild(this._body);
                var i = 0;
                for(i = 0; i < this._config.doubleSlider.markSeries.length; i++) {
                    var series = this._config.doubleSlider.markSeries[i];
                    series.index = i;
                    series.type = series.type || MarkType.LifeCycleEvent;
                    this._config.header.legendConfig.data.push({
                        text: series.label,
                        type: series.type
                    });
                }
                this._config.header.titleConfig.description = RulerUtilities.formatTitleTime(this._config.doubleSlider.timeRange.elapsed, UnitFormat.fullName);
                var header = new RulerHeader(this._config.header);
                header.render();
                this._doubleSlider = new DoubleSlider(this._config.doubleSlider);
                this._doubleSlider.render();
            };
            Ruler.prototype.addMark = function (id, markData) {
                if(this._doubleSlider) {
                    this._doubleSlider.addMark(id, markData);
                }
            };
            return Ruler;
        })();
        Graph.Ruler = Ruler;        
        var DoubleSlider = (function () {
            function DoubleSlider(config) {
                this._containerOffsetWidth = null;
                this._focused = false;
                this._rulerUnselectedCss = "ruler-unselected";
                this._rulerBodyCss = "ruler-body";
                this._isZoomed = false;
                this._selectionPixelStartX = 0;
                this._selectionPixelWidth = 0;
                this._onResizeHandler = this.onResize.bind(this);
                if(config && config.containerId) {
                    this._container = document.getElementById(config.containerId);
                    if(this._container) {
                        this._config = config;
                        this.initialize();
                    } else {
                        throw new Error(Plugin.Resources.getErrorString("JSPerf.1051"));
                    }
                } else {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1050"));
                }
            }
            DoubleSlider.prototype.deinitialize = function () {
                window.removeEventListener("resize", this._onResizeHandler);
                if(this._scale) {
                    this._scale.deinitialize();
                }
            };
            Object.defineProperty(DoubleSlider.prototype, "containerOffsetWidth", {
                get: function () {
                    if(this._containerOffsetWidth === null) {
                        this._containerOffsetWidth = this._container.offsetWidth;
                    }
                    return this._containerOffsetWidth;
                },
                enumerable: true,
                configurable: true
            });
            DoubleSlider.prototype.initialize = function () {
                this._publisher = new Common.Publisher(this._config.events);
                window.addEventListener("resize", this._onResizeHandler);
                this._currentTimeRange = this._config.timeRange;
                this._originalTimeRange = this._currentTimeRange;
                this._handlePosition = this._currentTimeRange;
                this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus());
                this._tabFocus = this.onTabFocus.bind(this);
                this._tabBlur = this.onTabBlur.bind(this);
                this._keyDown = this.onKeyDown.bind(this);
                this._keyUp = this.onKeyUp.bind(this);
                this._mouseMoveListener = this.onMouseMove.bind(this);
                this._mouseUpListener = this.onMouseUp.bind(this);
                this._startSelection = this.onStartSelection.bind(this);
                this._onDrag = this.onDrag.bind(this);
                this._container.addEventListener("mousedown", this._startSelection);
            };
            DoubleSlider.prototype.raiseFilterEvent = function (position, commandsStatus, isIntermittent) {
                if (typeof isIntermittent === "undefined") { isIntermittent = false; }
                var args = {
                    type: RulerEvents.Filter,
                    data: {
                        position: position,
                        isIntermittent: isIntermittent,
                        commandsStatus: commandsStatus
                    }
                };
                this._publisher.invokeListener(args);
            };
            DoubleSlider.prototype.raiseZoomEvent = function (selection, commandsStatus) {
                var args = {
                    type: RulerEvents.Zoom,
                    data: {
                        selection: selection,
                        commandsStatus: commandsStatus
                    }
                };
                this._publisher.invokeListener(args);
            };
            DoubleSlider.prototype.getCommandsStatus = function () {
                return {
                    isZoomInEnabled: this.getZoomInButtonState(),
                    isResetZoomEnabled: this.getResetZoomButtonState(),
                    isClearSelectionEnabled: this.getClearSelectionButtonState()
                };
            };
            DoubleSlider.prototype.getZoomInButtonState = function () {
                var isEnabled = true;
                if(this._handlePosition.elapsed.msec < this._config.minimumZoomLevelMsec || this._handlePosition.equals(this._currentTimeRange)) {
                    isEnabled = false;
                }
                return isEnabled;
            };
            DoubleSlider.prototype.getResetZoomButtonState = function () {
                var isEnabled = false;
                if(!this._originalTimeRange.equals(this._currentTimeRange)) {
                    isEnabled = true;
                }
                return isEnabled;
            };
            DoubleSlider.prototype.getClearSelectionButtonState = function () {
                var isEnabled = false;
                if(!this._handlePosition.equals(this._currentTimeRange)) {
                    isEnabled = true;
                }
                return isEnabled;
            };
            DoubleSlider.prototype.renderRulerBar = function () {
                this._bar = document.createElement("div");
                this._bar.id = this._config.bar.id;
                this._bar.className = this._config.bar.className;
                this._bar.style.left = (this._config.bar.left) + "px";
                this._bar.style.width = (this.containerOffsetWidth - this._config.bar.left - this._config.bar.right) + "px";
                this._container.appendChild(this._bar);
            };
            DoubleSlider.prototype.renderRulerScale = function () {
                var config = {
                    timeRange: this._currentTimeRange,
                    containerId: this._bar.id,
                    series: this._config.markSeries
                };
                this._scale = new VisualProfiler.Graph.RulerScale(config);
                this._scale.render();
                this.setAriaLabelForRuler();
                this.setAriaLabelForSliders();
            };
            DoubleSlider.prototype.renderLeftSliderHandle = function () {
                this._lSlider = document.createElement("div");
                this._lSlider.className = this._config.leftSlider.className;
                this._lSlider.style.width = this._config.leftSlider.width + "px";
                var sliderLeft = this._config.bar.left - this._config.leftSlider.width;
                this._lSlider.style.left = sliderLeft + "px";
                this._lSlider.tabIndex = 0;
                this._container.appendChild(this._lSlider);
                this._config.leftSlider.left = sliderLeft;
                this._lSlider.addEventListener("mousedown", this.mouseDown.bind(this));
                this._lSlider.addEventListener("focus", this._tabFocus);
                this._lSlider.addEventListener("blur", this._tabBlur);
                this.setAriaLabelForLeftSlider();
                this._lUnselectedRegion = document.createElement("div");
                this._lUnselectedRegion.className = this._rulerUnselectedCss;
                this._lUnselectedRegion.style.width = "0px";
                this._lUnselectedRegion.style.left = this._config.bar.left + "px";
                this._container.appendChild(this._lUnselectedRegion);
            };
            DoubleSlider.prototype.setAriaLabelForLeftSlider = function () {
                if(this._lSlider) {
                    this._lSlider.setAttribute("role", "slider");
                    this._lSlider.setAttribute("aria-label", Plugin.Resources.getString("RulerLeftSliderAriaLabel"));
                    var label = Plugin.Resources.getString("RulerSliderAriaValueText", RulerUtilities.formatTime(this._handlePosition.begin, UnitFormat.fullName), RulerUtilities.formatTime(this._currentTimeRange.begin, UnitFormat.fullName), RulerUtilities.formatTime(this._handlePosition.end, UnitFormat.fullName));
                    this._lSlider.setAttribute("aria-valuetext", label);
                    this._lSlider.setAttribute("aria-valuenow", this._handlePosition.begin.sec.toString());
                    this._lSlider.setAttribute("aria-valuemin", this._currentTimeRange.begin.sec.toString());
                    this._lSlider.setAttribute("aria-valuemax", this._handlePosition.end.sec.toString());
                }
            };
            DoubleSlider.prototype.setAriaLabelForRightSlider = function () {
                if(this._rSlider) {
                    this._rSlider.setAttribute("role", "slider");
                    this._rSlider.setAttribute("aria-label", Plugin.Resources.getString("RulerRightSliderAriaLabel"));
                    var label = Plugin.Resources.getString("RulerSliderAriaValueText", RulerUtilities.formatTime(this._handlePosition.end, UnitFormat.fullName), RulerUtilities.formatTime(this._handlePosition.begin, UnitFormat.fullName), RulerUtilities.formatTime(this._currentTimeRange.end, UnitFormat.fullName));
                    this._rSlider.setAttribute("aria-valuetext", label);
                    this._rSlider.setAttribute("aria-valuenow", this._handlePosition.end.sec.toString());
                    this._rSlider.setAttribute("aria-valuemin", this._handlePosition.begin.sec.toString());
                    this._rSlider.setAttribute("aria-valuemax", this._currentTimeRange.end.sec.toString());
                }
            };
            DoubleSlider.prototype.setAriaLabelForSliders = function () {
                this.setAriaLabelForLeftSlider();
                this.setAriaLabelForRightSlider();
            };
            DoubleSlider.prototype.renderRightSliderHandle = function () {
                this._rSlider = document.createElement("div");
                this._rSlider.className = this._config.rightSlider.className;
                this._rSlider.style.width = this._config.rightSlider.width + "px";
                var sliderLeft = this.containerOffsetWidth - this._config.bar.right;
                this._rSlider.style.left = sliderLeft + "px";
                this._rSlider.tabIndex = 0;
                this._container.appendChild(this._rSlider);
                this._config.rightSlider.left = sliderLeft;
                this._rSlider.addEventListener("mousedown", this.mouseDown.bind(this));
                this._rSlider.addEventListener("focus", this._tabFocus);
                this._rSlider.addEventListener("blur", this._tabBlur);
                this.setAriaLabelForRightSlider();
                this._rUnselectedRegion = document.createElement("div");
                this._rUnselectedRegion.className = this._rulerUnselectedCss;
                this._rUnselectedRegion.style.width = "0px";
                this._rUnselectedRegion.style.left = sliderLeft + "px";
                this._container.appendChild(this._rUnselectedRegion);
            };
            DoubleSlider.prototype.onResize = function (e) {
                this.invalidateSizeCache();
                var newRulerBarWidth = this.containerOffsetWidth - this._config.bar.left - this._config.bar.right;
                this._bar.style.width = newRulerBarWidth + "px";
                var lValue = this.convertToPixel(this._handlePosition.begin);
                var rValue = this.convertToPixel(this._handlePosition.end);
                var positionChanged = false;
                if(rValue - lValue < this._config.minimumRangeInPixel) {
                    if(lValue + this._config.minimumRangeInPixel <= newRulerBarWidth) {
                        rValue = lValue + this._config.minimumRangeInPixel;
                        positionChanged = true;
                    } else {
                        if(rValue - this._config.minimumRangeInPixel >= 0) {
                            lValue = rValue - this._config.minimumRangeInPixel;
                            positionChanged = true;
                        }
                    }
                }
                var lSliderLeft = lValue - this._config.leftSlider.width + this._config.bar.left;
                this._lSlider.style.left = lSliderLeft + "px";
                this._config.leftSlider.left = lSliderLeft;
                this._lUnselectedRegion.style.width = lValue + "px";
                var rSliderLeft = rValue + this._config.bar.left;
                this._rSlider.style.left = rSliderLeft + "px";
                this._config.rightSlider.left = rSliderLeft;
                this._rUnselectedRegion.style.left = rSliderLeft + "px";
                this._rUnselectedRegion.style.width = Math.ceil(this.containerOffsetWidth - rSliderLeft - this._config.bar.right) + "px";
                if(positionChanged) {
                    this._handlePosition = new VisualProfiler.TimeSpan(this.convertToTime(lValue), this.convertToTime(rValue));
                    this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus(), true);
                    this.setAriaLabelForSliders();
                }
            };
            DoubleSlider.prototype.onTabFocus = function (e) {
                if(e.srcElement === this._lSlider) {
                    this._currentSlider = this._lSlider;
                    this._lSlider.addEventListener("keydown", this._keyDown, false);
                    this._lSlider.addEventListener("keyup", this._keyUp, false);
                } else {
                    if(e.srcElement === this._rSlider) {
                        this._currentSlider = this._rSlider;
                        this._rSlider.addEventListener("keydown", this._keyDown, false);
                        this._rSlider.addEventListener("keyup", this._keyUp, false);
                    }
                }
                this._focused = true;
            };
            DoubleSlider.prototype.onTabBlur = function (e) {
                if(e.srcElement === this._lSlider) {
                    this._lSlider.removeEventListener("keydown", this._keyDown, false);
                    this._lSlider.removeEventListener("keyup", this._keyUp, false);
                } else {
                    if(e.srcElement === this._rSlider) {
                        this._rSlider.removeEventListener("keydown", this._keyDown, false);
                        this._rSlider.removeEventListener("keyup", this._keyUp, false);
                    }
                }
                this._currentSlider = null;
                this._focused = false;
            };
            DoubleSlider.prototype.onKeyDown = function (e) {
                this.handleKeyEvent(e, true);
            };
            DoubleSlider.prototype.onKeyUp = function (e) {
                this.handleKeyEvent(e, false);
            };
            DoubleSlider.prototype.handleKeyEvent = function (e, isDown) {
                var isIntermittent = isDown;
                if(this._focused && e.keyCode === KeyCodes.ARROW_LEFT) {
                    if(e.srcElement === this._lSlider) {
                        this.setHandlePosition(parseFloat(this._lSlider.style.left) - this._config.step, isIntermittent);
                    } else {
                        if(e.srcElement === this._rSlider) {
                            this.setHandlePosition(parseFloat(this._rSlider.style.left) - this._config.step, isIntermittent);
                        }
                    }
                }
                if(this._focused && e.keyCode === KeyCodes.ARROW_RIGHT) {
                    if(e.srcElement === this._lSlider) {
                        this.setHandlePosition(parseFloat(this._lSlider.style.left) + this._config.step, isIntermittent);
                    } else {
                        if(e.srcElement === this._rSlider) {
                            this.setHandlePosition(parseFloat(this._rSlider.style.left) + this._config.step, isIntermittent);
                        }
                    }
                }
            };
            DoubleSlider.prototype.mouseDown = function (e) {
                var rect = this._container.getBoundingClientRect();
                var root = document.documentElement;
                var x = e.clientX - rect.left - root.scrollLeft;
                if(x >= parseFloat(this._lSlider.style.left) && x <= parseFloat(this._lSlider.style.left) + parseFloat(this._lSlider.style.width)) {
                    this._currentSlider = this._lSlider;
                } else {
                    if(x >= parseFloat(this._rSlider.style.left) && x <= parseFloat(this._rSlider.style.left) + parseFloat(this._rSlider.style.width)) {
                        this._currentSlider = this._rSlider;
                    }
                }
                if(this._currentSlider) {
                    document.addEventListener("mousemove", this._mouseMoveListener, false);
                    document.addEventListener("mouseup", this._mouseUpListener, false);
                }
                return false;
            };
            DoubleSlider.prototype.setHandlePosition = function (position, isIntermittent) {
                if(this._currentSlider) {
                    var handle;
                    if(this._currentSlider === this._lSlider && (parseFloat(this._rSlider.style.left) - position - this._config.leftSlider.width) >= this._config.minimumRangeInPixel) {
                        if(position >= this._config.bar.left - this._config.leftSlider.width && position <= parseFloat(this._rSlider.style.left) - parseFloat(this._lSlider.style.width)) {
                            this._currentSlider.style.left = position + "px";
                            this._lUnselectedRegion.style.width = position - this._config.bar.left + "px";
                            this._config.leftSlider.left = position;
                            handle = new VisualProfiler.TimeSpan(this.convertToTime(position + this._config.leftSlider.width - this._config.bar.left), this._handlePosition.end);
                        } else {
                            if(position < this._config.bar.left) {
                                this._currentSlider.style.left = this._config.bar.left - this._config.leftSlider.width + "px";
                                this._lUnselectedRegion.style.width = "0px";
                                this._config.leftSlider.left = this._config.bar.left - this._config.leftSlider.width;
                                handle = new VisualProfiler.TimeSpan(this._currentTimeRange.begin, this._handlePosition.end);
                            }
                        }
                    } else {
                        if(this._currentSlider === this._rSlider && (position - parseFloat(this._lSlider.style.left) - this._config.leftSlider.width) >= this._config.minimumRangeInPixel) {
                            if(position <= this.containerOffsetWidth - this._config.bar.right && position >= parseFloat(this._lSlider.style.left) + parseFloat(this._lSlider.style.width)) {
                                this._currentSlider.style.left = position + "px";
                                this._rUnselectedRegion.style.left = position + "px";
                                this._rUnselectedRegion.style.width = Math.ceil(this.containerOffsetWidth - this._config.bar.right - position) + "px";
                                this._config.rightSlider.left = position;
                                handle = new VisualProfiler.TimeSpan(this._handlePosition.begin, this.convertToTime(position - this._config.bar.left));
                            } else {
                                if(position > this.containerOffsetWidth - this._config.bar.right) {
                                    this._currentSlider.style.left = this.containerOffsetWidth - this._config.bar.right + "px";
                                    this._rUnselectedRegion.style.left = this.containerOffsetWidth - this._config.bar.right + "px";
                                    this._rUnselectedRegion.style.width = "0px";
                                    this._config.rightSlider.left = this.containerOffsetWidth - this._config.bar.right;
                                    handle = new VisualProfiler.TimeSpan(this._handlePosition.begin, this._currentTimeRange.end);
                                }
                            }
                        }
                    }
                    if(handle) {
                        this._handlePosition = handle;
                        this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus(), isIntermittent);
                    } else {
                        if(!isIntermittent && this._handlePosition) {
                            this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus(), isIntermittent);
                        }
                    }
                    this.setAriaLabelForSliders();
                }
            };
            DoubleSlider.prototype.convertToTime = function (pixels) {
                var time = 0;
                if(this.containerOffsetWidth - this._config.bar.left - this._config.bar.right > 0) {
                    time = this._currentTimeRange.begin.nsec + (pixels * this._currentTimeRange.elapsed.nsec) / (this.containerOffsetWidth - this._config.bar.left - this._config.bar.right);
                }
                return new VisualProfiler.TimeStamp(time);
            };
            DoubleSlider.prototype.onMouseMove = function (e) {
                this.handleMouseEvent(e, true);
            };
            DoubleSlider.prototype.onMouseUp = function (e) {
                this.handleMouseEvent(e, false);
                document.removeEventListener("mousemove", this._mouseMoveListener, false);
                document.removeEventListener("mouseup", this._mouseUpListener, false);
                if(!this._focused) {
                    this._currentSlider = null;
                }
            };
            DoubleSlider.prototype.handleMouseEvent = function (e, isMove) {
                var isIntermittent = isMove;
                var rect = this._container.getBoundingClientRect();
                var root = document.documentElement;
                var mouseX = e.clientX - rect.left - root.scrollLeft;
                if(this._currentSlider && (mouseX < parseFloat(this._currentSlider.style.left) || mouseX > parseFloat(this._currentSlider.style.left) + parseFloat(this._currentSlider.style.width))) {
                    this.setHandlePosition(mouseX, isIntermittent);
                }
            };
            DoubleSlider.getDragDirection = function getDragDirection(selectionWidth) {
                if(selectionWidth > 0) {
                    return VisualProfiler.DragDirection.right;
                } else {
                    if(selectionWidth < 0) {
                        return VisualProfiler.DragDirection.left;
                    }
                }
                return VisualProfiler.DragDirection.none;
            }
            DoubleSlider.prototype.checkIfMouseInsideValidSelectionArea = function (event) {
                var isValid = false;
                var rect = this._container.getBoundingClientRect();
                var mouseX = event.clientX - rect.left - this._bar.offsetLeft;
                if(mouseX >= 0 && mouseX <= this._bar.clientWidth) {
                    isValid = true;
                }
                return isValid;
            };
            DoubleSlider.prototype.getSelectionStartWidth = function (event) {
                var rect = this._container.getBoundingClientRect();
                var mouseX = event.clientX - rect.left - this._bar.offsetLeft;
                return mouseX;
            };
            DoubleSlider.prototype.drawSelection = function () {
                this._lUnselectedRegion.style.width = this._selectionPixelStartX + "px";
                this._lSlider.style.left = this._selectionPixelStartX - this._config.leftSlider.width + this._bar.offsetLeft + "px";
                this._config.leftSlider.left = this._selectionPixelStartX + this._bar.offsetLeft;
                var left = this._selectionPixelStartX + this._selectionPixelWidth + this._bar.offsetLeft;
                this._rUnselectedRegion.style.left = left + "px";
                this._rUnselectedRegion.style.width = (this._bar.clientWidth - this._selectionPixelStartX - this._selectionPixelWidth) + "px";
                this._config.rightSlider.left = left;
                this._rSlider.style.left = left + "px";
                if(this._dragDirection === VisualProfiler.DragDirection.left) {
                    this._lSlider.focus();
                } else {
                    this._rSlider.focus();
                }
                this._handlePosition = new VisualProfiler.TimeSpan(this.convertToTime(this._selectionPixelStartX), this.convertToTime(this._selectionPixelStartX + this._selectionPixelWidth));
            };
            DoubleSlider.prototype.ensureMinSelectionWidth = function () {
                if(typeof this._selectionPixelStartX === "undefined") {
                    return;
                }
                if(Math.abs(this._selectionPixelWidth) < this._config.minimumRangeInPixel) {
                    this._selectionPixelWidth = this._config.minimumRangeInPixel;
                    switch(this._dragDirection) {
                        case VisualProfiler.DragDirection.right: {
                            if(this._selectionPixelStartX + this._selectionPixelWidth > this._bar.clientWidth) {
                                this._selectionPixelStartX = this._bar.clientWidth - this._selectionPixelWidth;
                            }
                            break;

                        }
                        case VisualProfiler.DragDirection.left: {
                            if(this._selectionPixelStartX - this._selectionPixelWidth < 0) {
                                this._selectionPixelStartX = 0;
                            } else {
                                if(this._selectionPixelStartX + this._selectionPixelWidth > this._bar.clientWidth) {
                                    this._selectionPixelStartX = this._bar.clientWidth - this._selectionPixelWidth;
                                } else {
                                    this._selectionPixelStartX = this._initialSelectionPixelStartX - this._selectionPixelWidth;
                                }
                            }
                            break;

                        }
                        default: {
                            if(this._selectionPixelStartX + (this._selectionPixelWidth / 2) > this._bar.clientWidth) {
                                this._selectionPixelStartX = this._bar.clientWidth - this._selectionPixelWidth;
                            } else {
                                if(this._selectionPixelStartX - (this._selectionPixelWidth / 2) < 0) {
                                    this._selectionPixelStartX = 0;
                                } else {
                                    this._selectionPixelStartX = this._selectionPixelStartX - (this._selectionPixelWidth / 2);
                                }
                            }
                            break;

                        }
                    }
                }
            };
            DoubleSlider.prototype.onStartSelection = function (event) {
                if((event.target !== this._lSlider && event.target !== this._rSlider) && this.checkIfMouseInsideValidSelectionArea(event)) {
                    this._initialSelectionPixelStartX = this._selectionPixelStartX = this.getSelectionStartWidth(event);
                    this._selectionPixelWidth = 0;
                    this._dragDirection = DoubleSlider.getDragDirection(this._selectionPixelWidth);
                    this._mouseupHandler = this.onStopSelection.bind(this);
                    this._container.addEventListener("mousemove", this._onDrag);
                    this._container.addEventListener("mouseup", this._mouseupHandler);
                    this._container.setCapture(true);
                    this.drawSelection();
                }
            };
            DoubleSlider.prototype.onStopSelection = function (event) {
                this._container.removeEventListener("mousemove", this._onDrag);
                this._container.removeEventListener("mouseup", this._mouseupHandler);
                this._container.releaseCapture();
                this.ensureMinSelectionWidth();
                this.drawSelection();
                this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus());
                this.setAriaLabelForSliders();
                this._dragDirection = VisualProfiler.DragDirection.none;
            };
            DoubleSlider.prototype.onDrag = function (event) {
                if(this.checkIfMouseInsideValidSelectionArea(event)) {
                    var xPixels = this.getSelectionStartWidth(event);
                    this._dragDirection = DoubleSlider.getDragDirection(xPixels - this._initialSelectionPixelStartX);
                    this._selectionPixelWidth = Math.abs(xPixels - this._initialSelectionPixelStartX);
                    if(this._dragDirection === VisualProfiler.DragDirection.left) {
                        this._selectionPixelStartX = this._initialSelectionPixelStartX - this._selectionPixelWidth;
                    }
                    this.drawSelection();
                    this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus());
                }
            };
            DoubleSlider.prototype.resetHandlePosition = function () {
                this._lSlider.style.left = (this._config.bar.left - this._config.leftSlider.width) + "px";
                this._config.leftSlider.left = (this._config.bar.left - this._config.leftSlider.width);
                this._lUnselectedRegion.style.width = "0px";
                this._lUnselectedRegion.style.left = this._config.bar.left + "px";
                var rSliderLeft = this.containerOffsetWidth - this._config.bar.right;
                this._rSlider.style.left = rSliderLeft + "px";
                this._config.rightSlider.left = rSliderLeft;
                this._rUnselectedRegion.style.width = "0px";
                this._rUnselectedRegion.style.left = rSliderLeft + "px";
            };
            DoubleSlider.prototype.resetScale = function () {
                this._scale.timeRange = this._currentTimeRange;
                this._scale.render();
                this.setAriaLabelForRuler();
                this.setAriaLabelForSliders();
            };
            DoubleSlider.prototype.convertToPixel = function (time) {
                var pixels = 0;
                if(this._currentTimeRange.elapsed.nsec > 0) {
                    pixels = (time.nsec - this._currentTimeRange.begin.nsec) * (this.containerOffsetWidth - this._config.bar.left - this._config.bar.right) / this._currentTimeRange.elapsed.nsec;
                }
                return pixels;
            };
            DoubleSlider.prototype.setAriaLabelForRuler = function () {
                if(this._container) {
                    var label = Plugin.Resources.getString("RulerAriaLabel", RulerUtilities.formatTime(this._currentTimeRange.begin, UnitFormat.fullName), RulerUtilities.formatTime(this._currentTimeRange.end, UnitFormat.fullName));
                    this._container.setAttribute("aria-label", label);
                }
            };
            DoubleSlider.prototype.addEventListener = function (eventType, func) {
                this._publisher.addEventListener(eventType, func);
            };
            DoubleSlider.prototype.removeEventListener = function (eventType, func) {
                this._publisher.removeEventListener(eventType, func);
            };
            DoubleSlider.prototype.clearSelection = function () {
                this._handlePosition = this._currentTimeRange;
                this.resetHandlePosition();
                var commandsStatus = this.getCommandsStatus();
                this.raiseZoomEvent(this._currentTimeRange, commandsStatus);
                this.raiseFilterEvent(this._handlePosition, commandsStatus);
                this.setAriaLabelForSliders();
            };
            Object.defineProperty(DoubleSlider.prototype, "SliderHandlePosition", {
                set: function (position) {
                    if(position && (this._currentTimeRange.begin.nsec <= position.begin.nsec && this._currentTimeRange.end.nsec >= position.end.nsec) && (!this._handlePosition || !this._handlePosition.equals(position))) {
                        this._handlePosition = position;
                        var lSliderLeft = this.convertToPixel(this._handlePosition.begin) + this._config.bar.left - this._config.leftSlider.width;
                        this._lSlider.style.left = lSliderLeft + "px";
                        this._config.leftSlider.left = lSliderLeft;
                        this._lUnselectedRegion.style.width = lSliderLeft - this._config.bar.left + this._config.leftSlider.width + "px";
                        this._lUnselectedRegion.style.left = this._config.bar.left + "px";
                        var rSliderLeft = this.convertToPixel(this._handlePosition.end) + this._config.bar.left;
                        this._rSlider.style.left = rSliderLeft + "px";
                        this._config.rightSlider.left = rSliderLeft;
                        this._rUnselectedRegion.style.width = Math.ceil(this.containerOffsetWidth - rSliderLeft - this._config.bar.right) + "px";
                        this._rUnselectedRegion.style.left = rSliderLeft + "px";
                        this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus());
                        this.setAriaLabelForSliders();
                    }
                },
                enumerable: true,
                configurable: true
            });
            DoubleSlider.prototype.setTimeRange = function (timespan) {
                this._currentTimeRange = timespan;
                this.resetScale();
            };
            DoubleSlider.prototype.onZoomIn = function () {
                if(this._handlePosition.elapsed.msec >= this._config.minimumZoomLevelMsec) {
                    this._currentTimeRange = this._handlePosition;
                    this.resetHandlePosition();
                    this.resetScale();
                    this.raiseZoomEvent(this._currentTimeRange, this.getCommandsStatus());
                    this.setAriaLabelForRuler();
                    this.setAriaLabelForSliders();
                }
            };
            DoubleSlider.prototype.onResetZoom = function () {
                this._currentTimeRange = this._originalTimeRange;
                var newRulerBarWidth = this.containerOffsetWidth - this._config.bar.left - this._config.bar.right;
                var lValue = this.convertToPixel(this._handlePosition.begin);
                var rValue = this.convertToPixel(this._handlePosition.end);
                if(rValue - lValue < this._config.minimumRangeInPixel) {
                    var isNew = false;
                    if(lValue + this._config.minimumRangeInPixel <= newRulerBarWidth) {
                        rValue = lValue + this._config.minimumRangeInPixel;
                        isNew = true;
                    } else {
                        if(rValue - this._config.minimumRangeInPixel >= 0) {
                            lValue = rValue - this._config.minimumRangeInPixel;
                            isNew = true;
                        }
                    }
                    if(isNew) {
                        this._handlePosition = new VisualProfiler.TimeSpan(this.convertToTime(lValue), this.convertToTime(rValue));
                        this.raiseFilterEvent(this._handlePosition, this.getCommandsStatus());
                    }
                }
                var lSliderLeft = lValue - this._config.leftSlider.width + this._config.bar.left;
                this._lSlider.style.left = lSliderLeft + "px";
                this._config.leftSlider.left = lSliderLeft;
                this._lUnselectedRegion.style.width = lValue + "px";
                var rSliderLeft = rValue + this._config.bar.left;
                this._rSlider.style.left = rSliderLeft + "px";
                this._config.rightSlider.left = rSliderLeft;
                this._rUnselectedRegion.style.left = rSliderLeft + "px";
                this._rUnselectedRegion.style.width = Math.ceil(this.containerOffsetWidth - rSliderLeft - this._config.bar.right) + "px";
                this.resetScale();
                this.raiseZoomEvent(this._currentTimeRange, this.getCommandsStatus());
                this.setAriaLabelForRuler();
                this.setAriaLabelForSliders();
            };
            DoubleSlider.prototype.invalidateSizeCache = function () {
                this._containerOffsetWidth = null;
                if(this._scale) {
                    this._scale.invalidateSizeCache();
                }
            };
            DoubleSlider.prototype.render = function () {
                while(this._container.hasChildNodes()) {
                    this._container.removeChild(this._container.firstChild);
                }
                this._container.className = this._config.className = this._rulerBodyCss;
                this._containerWidth = this.containerOffsetWidth;
                this._container.tabIndex = 0;
                this.renderRulerBar();
                this.renderRulerScale();
                this.renderLeftSliderHandle();
                this.renderRightSliderHandle();
                this.setAriaLabelForRuler();
            };
            DoubleSlider.prototype.addMark = function (id, markData) {
                if(this._scale) {
                    this._scale.addMark(id, markData);
                }
            };
            return DoubleSlider;
        })();
        Graph.DoubleSlider = DoubleSlider;        
        var MarkData = (function () {
            function MarkData(time, tooltipCallback) {
                this.time = time;
                this.getTooltipContent = tooltipCallback;
            }
            return MarkData;
        })();
        Graph.MarkData = MarkData;        
        var AggregatedMarkData = (function () {
            function AggregatedMarkData() {
                this._contentUpdateNeeded = true;
                this.marks = [];
            }
            AggregatedMarkData.prototype.updateData = function () {
                this._content = [];
                for(var i = 0; i < this.marks.length; i++) {
                    if(this.marks[i].getTooltipContent && typeof this.marks[i].getTooltipContent() !== "undefined") {
                        this._content.push(this.marks[i].getTooltipContent());
                    }
                }
                this._contentUpdateNeeded = false;
            };
            AggregatedMarkData.prototype.getTooltipContent = function () {
                if(this._contentUpdateNeeded) {
                    this.updateData();
                }
                return this._content.join("<br/>");
            };
            AggregatedMarkData.prototype.getAriaContent = function () {
                if(this._contentUpdateNeeded) {
                    this.updateData();
                }
                return this._content.join(", ");
            };
            AggregatedMarkData.prototype.length = function () {
                return this.marks.length;
            };
            AggregatedMarkData.prototype.push = function (mark) {
                if(!this.time) {
                    this.time = mark.time;
                }
                this.marks.push(mark);
                this._contentUpdateNeeded = true;
            };
            return AggregatedMarkData;
        })();
        Graph.AggregatedMarkData = AggregatedMarkData;        
        var MarkSeries = (function () {
            function MarkSeries() { }
            return MarkSeries;
        })();
        Graph.MarkSeries = MarkSeries;        
        var RulerScale = (function () {
            function RulerScale(config) {
                this._containerOffsetWidth = null;
                this._scaleMainDivClientWidth = null;
                this._scaleMarksImageHeight = 4;
                this._rulerScaleTickmarkBigCss = "ruler-scale-tickmark-big";
                this._rulerScaleTickmarkMediumCss = "ruler-scale-tickmark-medium";
                this._rulerScaleTickmarkSmallCss = "ruler-scale-tickmark-small";
                this._rulerScaleMarkImageAppEventCss = "ruler-scale-mark-image-appevent";
                this._rulerScaleMarkImageUserMarkCss = "ruler-scale-mark-image-usermark";
                this._rulerScaleLabelTextCss = "ruler-scale-label-text";
                this._rulerScaleMainCss = "ruler-scale-main";
                this._rulerScaleTickmarksCss = "ruler-scale-tickmarks";
                this._rulerScaleMarkImagePositionCss = "ruler-scale-mark-image-position";
                this._scaleMainDivBorderWidth = 1;
                this._minimumMarkDistance = 5;
                this._onResizeHandler = this.onResize.bind(this);
                if(config && config.containerId && config.timeRange) {
                    this._timeRange = config.timeRange;
                    this._container = document.getElementById(config.containerId);
                    this._container.onresize = this._onResizeHandler;
                    this._seriesList = config.series;
                    while(this._container.hasChildNodes()) {
                        this._container.removeChild(this._container.firstChild);
                    }
                } else {
                    new Error(Plugin.Resources.getErrorString("JSPerf.1050"));
                }
            }
            RulerScale.prototype.deinitialize = function () {
                window.removeEventListener("resize", this._onResizeHandler);
                if(this._container) {
                    this._container.onresize = null;
                }
            };
            Object.defineProperty(RulerScale.prototype, "containerOffsetWidth", {
                get: function () {
                    if(this._containerOffsetWidth === null) {
                        this._containerOffsetWidth = this._container.offsetWidth;
                    }
                    return this._containerOffsetWidth;
                },
                enumerable: true,
                configurable: true
            });
            Object.defineProperty(RulerScale.prototype, "scaleMainDivClientWidth", {
                get: function () {
                    if(this._scaleMainDivClientWidth === null) {
                        this._scaleMainDivClientWidth = this._scaleMainDiv.clientWidth;
                    }
                    return this._scaleMainDivClientWidth;
                },
                enumerable: true,
                configurable: true
            });
            RulerScale.prototype.renderTickMarks = function () {
                this._scaleBigTickMarkElementsFactory.start();
                this._scaleMediumTickMarkElementsFactory.start();
                this._scaleSmallTickMarkElementsFactory.start();
                this._scaleTickMarkLabelElementsFactory.start();
                for(var i = 0; i < this._tickMarkList.length; i++) {
                    var tick;
                    switch(this._tickMarkList[i].type) {
                        case TickMarkType.Big: {
                            tick = this._scaleBigTickMarkElementsFactory.getNext();
                            break;

                        }
                        case TickMarkType.Medium: {
                            tick = this._scaleMediumTickMarkElementsFactory.getNext();
                            break;

                        }
                        case TickMarkType.Small: {
                            tick = this._scaleSmallTickMarkElementsFactory.getNext();
                            break;

                        }
                    }
                    if(tick) {
                        tick.style.left = this.calculateElementPosition(this._tickMarkList[i].value) + "px";
                        if(TickMarkType.Big === this._tickMarkList[i].type) {
                            var label = this._scaleTickMarkLabelElementsFactory.getNext();
                            label.innerText = this._tickMarkList[i].label;
                            label.style.left = parseFloat(tick.style.left) + 1 + "px";
                        }
                    }
                }
                this._scaleBigTickMarkElementsFactory.stop();
                this._scaleMediumTickMarkElementsFactory.stop();
                this._scaleSmallTickMarkElementsFactory.stop();
                this._scaleTickMarkLabelElementsFactory.stop();
                this.renderMarks();
            };
            RulerScale.prototype.renderMarks = function () {
                var _this = this;
                if(this._scaleTickMarksDiv) {
                    this.createAggregateMarkList();
                    this._scaleMarkAppEventElementsFactory.start();
                    this._scaleMarkUserElementsFactory.start();
                    if(this._data) {
                        for(var j = 0; j < this._data.length; j++) {
                            var mark = this._data[j];
                            if(mark && mark.time && mark.time.nsec >= this._timeRange.begin.nsec && mark.time.nsec <= this._timeRange.end.nsec) {
                                (function (mark) {
                                    mark.type = mark.type || MarkType.LifeCycleEvent;
                                    var markDiv;
                                    if(!mark.type || mark.type === MarkType.LifeCycleEvent) {
                                        markDiv = _this._scaleMarkAppEventElementsFactory.getNext();
                                    } else {
                                        markDiv = _this._scaleMarkUserElementsFactory.getNext();
                                    }
                                    markDiv.style.left = mark.pixelPosition + "px";
                                    markDiv.onmouseover = function () {
                                        return _this.showMarkTooltip(mark);
                                    };
                                    markDiv.onmouseout = Plugin.VS.Tooltip.dismiss;
                                    markDiv.onfocus = function () {
                                        markDiv.setAttribute("aria-label", Plugin.Resources.getString("RulerMarkLabel", mark.getAriaContent()));
                                    };
                                })(mark);
                            }
                        }
                    }
                    this._scaleMarkAppEventElementsFactory.stop();
                    this._scaleMarkUserElementsFactory.stop();
                }
            };
            RulerScale.prototype.createAggregateMarkList = function () {
                var seriesList = this._seriesList;
                this._data = [];
                if(this._seriesList) {
                    for(var i = 0; i < seriesList.length; i++) {
                        var seriesData = seriesList[i].data;
                        if(!seriesData) {
                            seriesData = [];
                        }
                        for(var j = 0; j < seriesData.length; j++) {
                            var dataPoint = seriesData[j];
                            this.createAggregatedMark(dataPoint, seriesList[i].type);
                        }
                    }
                }
            };
            RulerScale.prototype.createAggregatedMark = function (dataPoint, type) {
                var pixelPosition = this.calculateElementPosition(dataPoint.time) - this._scaleMarksImageHeight;
                var isNewPoint = true;
                var index = -1;
                for(var k = 0; k < this._data.length; k++) {
                    if(pixelPosition <= this._data[k].pixelPosition + this._minimumMarkDistance && pixelPosition >= this._data[k].pixelPosition - this._minimumMarkDistance) {
                        this._data[k].push(dataPoint);
                        if(this._data[k].type !== type) {
                            this._data[k].type = MarkType.UserMark;
                        }
                        isNewPoint = false;
                        index = k;
                        break;
                    }
                }
                if(isNewPoint) {
                    var data = new AggregatedMarkData();
                    data.push(dataPoint);
                    data.type = type;
                    data.pixelPosition = pixelPosition;
                    this._data.push(data);
                    index = this._data.length - 1;
                }
                return index;
            };
            RulerScale.prototype.showMarkTooltip = function (mark) {
                var toolTipContent = mark.getTooltipContent();
                if(toolTipContent) {
                    var config = {
                        content: toolTipContent
                    };
                    Plugin.VS.Tooltip.show(config);
                }
            };
            RulerScale.prototype.createMarkVisual = function (className) {
                var markDiv = document.createElement("div");
                markDiv.tabIndex = 0;
                markDiv.classList.add(className);
                markDiv.classList.add(this._rulerScaleMarkImagePositionCss);
                return markDiv;
            };
            RulerScale.prototype.calculateElementPosition = function (time) {
                return Math.round((this.scaleMainDivClientWidth * (time.nsec - this._timeRange.begin.nsec) / this._timeRange.elapsed.nsec));
            };
            RulerScale.prototype.onResize = function () {
                this.invalidateSizeCache();
                this.render();
            };
            Object.defineProperty(RulerScale.prototype, "timeRange", {
                set: function (time) {
                    this._timeRange = time;
                },
                enumerable: true,
                configurable: true
            });
            RulerScale.prototype.invalidateSizeCache = function () {
                this._containerOffsetWidth = null;
                this._scaleMainDivClientWidth = null;
            };
            RulerScale.prototype.render = function () {
                var _this = this;
                if(this._container) {
                    if(!this._scaleMainDiv) {
                        this._scaleMainDiv = document.createElement("div");
                        this._scaleMainDiv.className = this._rulerScaleMainCss;
                        window.addEventListener("resize", this._onResizeHandler);
                        this._container.appendChild(this._scaleMainDiv);
                    }
                    if(!this._scaleTickMarksDiv) {
                        this._scaleTickMarksDiv = document.createElement("div");
                        this._scaleTickMarksDiv.className = this._rulerScaleTickmarksCss;
                        this._scaleMainDiv.appendChild(this._scaleTickMarksDiv);
                        this._scaleBigTickMarkElementsFactory = VisualProfiler.ElementRecyclerFactory.forDivWithClass(this._scaleTickMarksDiv, this._rulerScaleTickmarkBigCss);
                        this._scaleMediumTickMarkElementsFactory = VisualProfiler.ElementRecyclerFactory.forDivWithClass(this._scaleTickMarksDiv, this._rulerScaleTickmarkMediumCss);
                        this._scaleSmallTickMarkElementsFactory = VisualProfiler.ElementRecyclerFactory.forDivWithClass(this._scaleTickMarksDiv, this._rulerScaleTickmarkSmallCss);
                        this._scaleTickMarkLabelElementsFactory = VisualProfiler.ElementRecyclerFactory.forDivWithClass(this._scaleTickMarksDiv, this._rulerScaleLabelTextCss);
                        this._scaleMarkAppEventElementsFactory = new VisualProfiler.ElementRecyclerFactory(this._scaleTickMarksDiv, function () {
                            return _this.createMarkVisual(_this._rulerScaleMarkImageAppEventCss);
                        });
                        this._scaleMarkUserElementsFactory = new VisualProfiler.ElementRecyclerFactory(this._scaleTickMarksDiv, function () {
                            return _this.createMarkVisual(_this._rulerScaleMarkImageUserMarkCss);
                        });
                    }
                    this._tickMarkList = RulerUtilities.getTickMarksPosition(this._timeRange, this.containerOffsetWidth);
                    this.renderTickMarks();
                }
            };
            RulerScale.prototype.addMark = function (id, markData) {
                if(this._seriesList && id && markData && markData.time && markData.time.nsec >= this._timeRange.begin.nsec && markData.time.nsec <= this._timeRange.end.nsec) {
                    var series;
                    for(var j = 0; j < this._seriesList.length; j++) {
                        if(this._seriesList[j].id === id) {
                            series = this._seriesList[j];
                            if(!series.data) {
                                series.data = [];
                            }
                            series.data.push(markData);
                            break;
                        }
                    }
                    this.renderMarks();
                }
            };
            return RulerScale;
        })();
        Graph.RulerScale = RulerScale;        
        (function (UnitFormat) {
            UnitFormat._map = [];
            UnitFormat._map[0] = "italizedAbbreviations";
            UnitFormat.italizedAbbreviations = 0;
            UnitFormat._map[1] = "fullName";
            UnitFormat.fullName = 1;
        })(Graph.UnitFormat || (Graph.UnitFormat = {}));
        var UnitFormat = Graph.UnitFormat;
        var RulerUtilities = (function () {
            function RulerUtilities() { }
            RulerUtilities.counter = 0;
            RulerUtilities.OneMillisecond = 1000000;
            RulerUtilities.OneSecond = 1000 * 1000000;
            RulerUtilities.OneMinute = 60 * 1000 * 1000000;
            RulerUtilities.getRandomNumber = function getRandomNumber() {
                return RulerUtilities.counter++;
            }
            RulerUtilities.getTickMarksPosition = function getTickMarksPosition(timeRange, width) {
                var rangeNsec = timeRange.elapsed.nsec;
                var beginNsec = timeRange.begin.nsec;
                var tickMarkList = [];
                var intervalDuration = Math.pow(10, Math.floor(Math.log(rangeNsec) / Math.LN10));
                var intervalWidth = (width / rangeNsec) * intervalDuration;
                if(intervalWidth < 100) {
                    if(intervalWidth < 25) {
                        intervalDuration *= 8;
                    } else {
                        if(intervalWidth < 50) {
                            intervalDuration *= 4;
                        } else {
                            if(intervalWidth < 100) {
                                intervalDuration *= 2;
                            }
                        }
                    }
                } else {
                    if(intervalWidth > 250) {
                        if(intervalWidth < 400) {
                            intervalDuration /= 2;
                        } else {
                            if(intervalWidth < 800) {
                                intervalDuration /= 4;
                            } else {
                                if(intervalWidth < 1600) {
                                    intervalDuration /= 8;
                                } else {
                                    intervalDuration /= 10;
                                }
                            }
                        }
                    }
                }
                var smallTickDuration = intervalDuration / 10;
                var mediumTickDuration = intervalDuration / 2;
                intervalWidth = (width / rangeNsec) * intervalDuration;
                if(intervalWidth < 130) {
                    smallTickDuration = intervalDuration / 5;
                }
                tickMarkList = RulerUtilities.generateTickMarks(timeRange, beginNsec - (beginNsec % intervalDuration), intervalDuration, mediumTickDuration, smallTickDuration);
                return tickMarkList;
            }
            RulerUtilities.getVerticalLinePositions = function getVerticalLinePositions(timeRange, width) {
                var positions = [];
                var marks = RulerUtilities.getTickMarksPosition(timeRange, width);
                for(var i = 0; i < marks.length; ++i) {
                    var mark = marks[i];
                    if(mark.type === TickMarkType.Big) {
                        var position = (mark.value.nsec - timeRange.begin.nsec) / timeRange.elapsed.nsec * 100;
                        positions.push(position);
                    }
                }
                return positions;
            }
            RulerUtilities.generateTickMarks = function generateTickMarks(timeRange, start, bigTick, mediumTick, step) {
                var tickMarkList = [];
                var beginNsec = timeRange.begin.nsec;
                var endNsec = timeRange.end.nsec;
                start = Math.round(VisualProfiler.TimeStamp.fromNanoseconds(start).nsec);
                bigTick = Math.round(VisualProfiler.TimeStamp.fromNanoseconds(bigTick).nsec);
                mediumTick = Math.round(VisualProfiler.TimeStamp.fromNanoseconds(mediumTick).nsec);
                step = Math.round(VisualProfiler.TimeStamp.fromNanoseconds(step).nsec);
                for(var i = start; i < endNsec; i += step) {
                    if(i > beginNsec) {
                        var tickMarkTime = VisualProfiler.TimeStamp.fromNanoseconds(i);
                        if(!isNaN(i % bigTick) && i % bigTick === 0) {
                            tickMarkList.push({
                                type: TickMarkType.Big,
                                value: tickMarkTime,
                                label: RulerUtilities.formatTime(tickMarkTime)
                            });
                        } else {
                            if(!isNaN(i % mediumTick) && i % mediumTick === 0) {
                                tickMarkList.push({
                                    type: TickMarkType.Medium,
                                    value: tickMarkTime,
                                    label: RulerUtilities.formatTime(tickMarkTime)
                                });
                            } else {
                                tickMarkList.push({
                                    type: TickMarkType.Small,
                                    value: tickMarkTime,
                                    label: RulerUtilities.formatTime(tickMarkTime)
                                });
                            }
                        }
                    }
                }
                return tickMarkList;
            }
            RulerUtilities.formatTime = function formatTime(value, unitFormat) {
                if (typeof unitFormat === "undefined") { unitFormat = UnitFormat.italizedAbbreviations; }
                var splitTime = RulerUtilities.getSplittedTime(value);
                var valueNs = Math.floor(value.nsec);
                var time = "0";
                var nf = RulerUtilities.getNumberFormat();
                if(splitTime.minutes) {
                    if(!splitTime.seconds && !splitTime.milliseconds && !splitTime.nanoseconds) {
                        time = splitTime.minString + ":00";
                    } else {
                        if(!splitTime.milliseconds && !splitTime.nanoseconds) {
                            time = splitTime.minString + ":" + splitTime.secString;
                        } else {
                            if(!splitTime.nanoseconds) {
                                splitTime.msString = splitTime.msString.replace(/0*$/, "");
                                time = splitTime.minString + ":" + splitTime.secString + nf.numberDecimalSeparator + splitTime.msString;
                            } else {
                                splitTime.nsString = splitTime.nsString.replace(/0*$/, "");
                                time = splitTime.minString + ":" + splitTime.secString + nf.numberDecimalSeparator + splitTime.msString + splitTime.nsString;
                            }
                        }
                    }
                } else {
                    if(splitTime.seconds) {
                        if(!splitTime.milliseconds && !splitTime.nanoseconds) {
                            time = splitTime.secString;
                        } else {
                            if(!splitTime.nanoseconds) {
                                splitTime.msString = splitTime.msString.replace(/0*$/, "");
                                time = splitTime.secString + nf.numberDecimalSeparator + splitTime.msString;
                            } else {
                                splitTime.nsString = splitTime.nsString.replace(/0*$/, "");
                                time = splitTime.secString + nf.numberDecimalSeparator + splitTime.msString + splitTime.nsString;
                            }
                        }
                    } else {
                        if(splitTime.milliseconds) {
                            if(!splitTime.nanoseconds) {
                                time = splitTime.msString;
                            } else {
                                splitTime.nsString = splitTime.nsString.replace(/0*$/, "");
                                time = splitTime.msString + nf.numberDecimalSeparator + splitTime.nsString;
                            }
                        } else {
                            if(splitTime.nanoseconds) {
                                splitTime.nsString = splitTime.nsString.replace(/0*$/, "");
                                time = "0" + nf.numberDecimalSeparator + splitTime.nsString;
                            }
                        }
                    }
                }
                var unit = RulerUtilities.getUnit(valueNs, unitFormat);
                return time + unit;
            }
            RulerUtilities.formatTitleTime = function formatTitleTime(value, unitFormat) {
                if (typeof unitFormat === "undefined") { unitFormat = UnitFormat.fullName; }
                var splitTime = RulerUtilities.getSplittedTime(value);
                var valueNs = Math.floor(value.nsec);
                var time = "0";
                var nf = RulerUtilities.getNumberFormat();
                if(splitTime.minutes) {
                    if(!splitTime.seconds) {
                        time = splitTime.minString + ":00";
                    } else {
                        time = splitTime.minString + ":" + splitTime.secString;
                    }
                } else {
                    if(splitTime.seconds) {
                        if(!splitTime.milliseconds) {
                            time = splitTime.secString;
                        } else {
                            splitTime.msString = splitTime.msString.replace(/0*$/, "");
                            time = splitTime.secString + nf.numberDecimalSeparator + splitTime.msString;
                        }
                    } else {
                        if(splitTime.milliseconds) {
                            if(!splitTime.nanoseconds) {
                                time = splitTime.msString;
                            } else {
                                splitTime.nsString = splitTime.nsString.replace(/0*$/, "");
                                time = splitTime.msString + nf.numberDecimalSeparator + splitTime.nsString;
                            }
                        } else {
                            if(splitTime.nanoseconds) {
                                splitTime.nsString = splitTime.nsString.replace(/0*$/, "");
                                time = "0" + nf.numberDecimalSeparator + splitTime.nsString;
                            }
                        }
                    }
                }
                var unit = RulerUtilities.getUnit(valueNs, unitFormat);
                return time + unit;
            }
            RulerUtilities.getUnit = function getUnit(valueNs, unitFormat) {
                var units = this.getUnits(unitFormat);
                var unit;
                if(valueNs < RulerUtilities.OneSecond) {
                    unit = units.milliseconds;
                } else {
                    if(valueNs < RulerUtilities.OneMinute) {
                        unit = units.seconds;
                    } else {
                        unit = units.minutes;
                    }
                }
                return unit;
            }
            RulerUtilities.getUnits = function getUnits(unitFormat) {
                var unitLabelFormat;
                if(unitFormat === UnitFormat.fullName) {
                    unitLabelFormat = {
                        milliseconds: " " + Plugin.Resources.getString("MillisecondsLabel"),
                        seconds: " " + Plugin.Resources.getString("SecondsLabel"),
                        minutes: " " + Plugin.Resources.getString("MinutesLabel")
                    };
                } else {
                    unitLabelFormat = {
                        milliseconds: Plugin.Resources.getString("MillisecondsAbbreviation"),
                        seconds: Plugin.Resources.getString("SecondsAbbreviation"),
                        minutes: Plugin.Resources.getString("MinutesAbbreviation")
                    };
                }
                return unitLabelFormat;
            }
            RulerUtilities.getSplittedTime = function getSplittedTime(value) {
                var valueNs = Math.floor(value.nsec);
                if(valueNs < 0) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1052"));
                }
                var nanoseconds = Math.floor(valueNs % RulerUtilities.OneMillisecond);
                var milliseconds = Math.floor((valueNs - nanoseconds) % RulerUtilities.OneSecond);
                var seconds = Math.floor((valueNs - nanoseconds - milliseconds) % RulerUtilities.OneMinute);
                var minutes = Math.floor(valueNs - nanoseconds - milliseconds - seconds);
                minutes = minutes / RulerUtilities.OneMinute;
                seconds = seconds / RulerUtilities.OneSecond;
                milliseconds = milliseconds / RulerUtilities.OneMillisecond;
                var ns = "";
                if(nanoseconds > 999) {
                    ns = nanoseconds.toString().substr(0, 3);
                }
                var ms = "";
                if(ns || milliseconds) {
                    var ms = milliseconds.toString();
                    if(seconds) {
                        while(ms.length < 3) {
                            ms = "0" + ms;
                        }
                    }
                }
                var sec = "";
                if(ns || ms || seconds) {
                    sec = seconds.toString();
                    if(minutes) {
                        while(sec.length < 2) {
                            sec = "0" + sec;
                        }
                    }
                }
                var min = "";
                if(minutes) {
                    min = minutes.toString();
                }
                return {
                    minutes: minutes,
                    seconds: seconds,
                    milliseconds: milliseconds,
                    nanoseconds: nanoseconds,
                    nsString: ns,
                    msString: ms,
                    secString: sec,
                    minString: min
                };
            }
            RulerUtilities.getNumberFormat = function getNumberFormat() {
                var nf = Plugin.VS.Culture.NumberFormat;
                if(!nf || nf.length === 0) {
                    nf = {
                        numberDecimalSeparator: "."
                    };
                }
                return nf;
            }
            return RulerUtilities;
        })();
        Graph.RulerUtilities = RulerUtilities;        
        var RulerLegend = (function () {
            function RulerLegend(config) {
                this._rulerLegendColorDivCss = "ruler-legend-color-div";
                this._rulerLegendTextDivCss = "ruler-legend-text-div";
                this._rulerLegendDivCss = "ruler-legend-div";
                this._rulerScaleLabelImageAppEventCss = "ruler-label-mark-image-appevent";
                this._rulerScaleLabelImageUserMarkCss = "ruler-label-mark-image-usermark";
                this._rulerScaleMarkImageUserMarkLabel = "User Marks";
                this._rulerScaleMarkImageAppEventLabel = "App Lifecycle Events";
                if(!config || !config.containerId) {
                    throw new Error(Plugin.Resources.getErrorString("JSPerf.1050"));
                }
                this._config = config;
            }
            RulerLegend.prototype.render = function () {
                this.renderLegendItem(this._rulerScaleLabelImageUserMarkCss, Plugin.Resources.getString("Ruler_UserMarks") || this._rulerScaleMarkImageUserMarkLabel);
                this.renderLegendItem(this._rulerScaleLabelImageAppEventCss, Plugin.Resources.getString("Ruler_LifeCycleEvents") || this._rulerScaleMarkImageAppEventLabel);
            };
            RulerLegend.prototype.renderLegendItem = function (className, label) {
                var legendColorDiv = document.createElement("div");
                legendColorDiv.className = this._rulerLegendColorDivCss;
                var colorDiv = document.createElement("div");
                colorDiv.className = className;
                legendColorDiv.appendChild(colorDiv);
                var legendTextDiv = document.createElement("div");
                legendTextDiv.className = this._rulerLegendTextDivCss;
                legendTextDiv.innerHTML = label;
                var individualLegendBox = document.createElement("div");
                individualLegendBox.className = this._rulerLegendDivCss;
                individualLegendBox.appendChild(legendColorDiv);
                individualLegendBox.appendChild(legendTextDiv);
                var container = document.getElementById(this._config.containerId);
                container.appendChild(individualLegendBox);
            };
            return RulerLegend;
        })();
        Graph.RulerLegend = RulerLegend;        
        var RulerHeader = (function () {
            function RulerHeader(config) {
                this._rulerTitleTextCss = "ruler-title-text";
                this._config = config;
            }
            RulerHeader.prototype.render = function () {
                var titleContainer = document.createElement("div");
                titleContainer.id = this._config.titleConfig.containerId;
                titleContainer.className = this._config.titleConfig.className;
                var titleText = document.createElement("div");
                titleText.className = this._rulerTitleTextCss;
                var text = Plugin.Resources.getString("RulerTitle") || this._config.titleConfig.text || "";
                if(this._config.titleConfig.description) {
                    text += ": " + this._config.titleConfig.description;
                }
                titleText.innerHTML = text;
                titleContainer.appendChild(titleText);
                var legendContainer = document.createElement("div");
                legendContainer.id = this._config.legendConfig.containerId;
                legendContainer.className = this._config.legendConfig.className;
                var container = document.getElementById(this._config.containerId);
                container.appendChild(titleContainer);
                container.appendChild(legendContainer);
                this.legend = new RulerLegend(this._config.legendConfig);
                this.legend.render();
            };
            return RulerHeader;
        })();
        Graph.RulerHeader = RulerHeader;        
    })(VisualProfiler.Graph || (VisualProfiler.Graph = {}));
    var Graph = VisualProfiler.Graph;
})(VisualProfiler || (VisualProfiler = {}));
//@ sourceMappingURL=Ruler.js.map

// SIG // Begin signature block
// SIG // MIIamQYJKoZIhvcNAQcCoIIaijCCGoYCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFPZ4W1ClxbWM
// SIG // tMBIxRXDCb2sU3M0oIIVgjCCBMMwggOroAMCAQICEzMA
// SIG // AAArOTJIwbLJSPMAAAAAACswDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTEyMDkwNDIx
// SIG // MTIzNFoXDTEzMTIwNDIxMTIzNFowgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpDMEY0LTMwODYtREVGODEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBAKa2MA4DZa5QWoZrhZ9IoR7JwO5eSQeF4HCWfL65
// SIG // X2JfBibTizm7GCKlLpKt2EuIOhqvm4OuyF45jMIyexZ4
// SIG // 7Tc4OvFi+2iCAmjs67tAirH+oSw2YmBwOWBiDvvGGDhv
// SIG // sJLWQA2Apg14izZrhoomFxj/sOtNurspE+ZcSI5wRjYm
// SIG // /jQ1qzTh99rYXOqZfTG3TR9X63zWlQ1mDB4OMhc+LNWA
// SIG // oc7r95iRAtzBX/04gPg5f11kyjdcO1FbXYVfzh4c+zS+
// SIG // X+UoVXBUnLjsfABVRlsomChWTOHxugkZloFIKjDI9zMg
// SIG // bOdpw7PUw07PMB431JhS1KkjRbKuXEFJT7RiaJMCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBSlGDNTP5VgoUMW747G
// SIG // r9Irup5Y0DAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQB+zLB75S++
// SIG // 51a1z3PbqlLRFjnGtM361/4eZbXnSPObRogFZmomhl7+
// SIG // h1jcxmOOOID0CEZ8K3OxDr9BqsvHqpSkN/BkOeHF1fnO
// SIG // B86r5CXwaa7URuL+ZjI815fFMiH67holoF4MQiwRMzqC
// SIG // g/3tHbO+zpGkkSVxuatysJ6v5M8AYolwqbhKUIzuLyJk
// SIG // pajmTWuVLBx57KejMdqQYJCkbv6TAg0/LCQNxmomgVGD
// SIG // ShC7dWNEqmkIxgPr4s8L7VY67O9ypwoM9ADTIrivInKz
// SIG // 58ScCyiggMrj4dc5ZjDnRhcY5/qC+lkLeryoDf4c/wOL
// SIG // Y7JNEgIjTy2zhYQ74qFH6M8VMIIE7DCCA9SgAwIBAgIT
// SIG // MwAAALARrwqL0Duf3QABAAAAsDANBgkqhkiG9w0BAQUF
// SIG // ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQDExpN
// SIG // aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xMzAx
// SIG // MjQyMjMzMzlaFw0xNDA0MjQyMjMzMzlaMIGDMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYD
// SIG // VQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggEiMA0G
// SIG // CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDor1yiIA34
// SIG // KHy8BXt/re7rdqwoUz8620B9s44z5lc/pVEVNFSlz7SL
// SIG // qT+oN+EtUO01Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Un
// SIG // m/P4LZ5BNisLQPu+O7q5XHWTFlJLyjPFN7Dz636o9UEV
// SIG // XAhlHSE38Cy6IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpO
// SIG // oPXJCiHiquMBNkf9L4JqgQP1qTXclFed+0vUDoLbOI8S
// SIG // /uPWenSIZOFixCUuKq6dGB8OHrbCryS0DlC83hyTXEmm
// SIG // ebW22875cHsoAYS4KinPv6kFBeHgD3FN/a1cI4Mp68fF
// SIG // SsjoJ4TTfsZDC5UABbFPZXHFAgMBAAGjggFgMIIBXDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGm
// SIG // WjNN2pgHgP+EHr6H+XIyQfIwUQYDVR0RBEowSKRGMEQx
// SIG // DTALBgNVBAsTBE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRm
// SIG // YWYwYjcxLWFkMzctNGFhMy1hNjcxLTc2YmMwNTIzNDRh
// SIG // ZDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWND
// SIG // b2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BD
// SIG // QV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOC
// SIG // AQEAMdduKhJXM4HVncbr+TrURE0Inu5e32pbt3nPApy8
// SIG // dmiekKGcC8N/oozxTbqVOfsN4OGb9F0kDxuNiBU6fNut
// SIG // zrPJbLo5LEV9JBFUJjANDf9H6gMH5eRmXSx7nR2pEPoc
// SIG // sHTyT2lrnqkkhNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2y
// SIG // RPnwPJNtQtjofOYXoJtoaAko+QKX7xEDumdSrcHps3Om
// SIG // 0mPNSuI+5PNO/f+h4LsCEztdIN5VP6OukEAxOHUoXgSp
// SIG // Rm3m9Xp5QL0fzehF1a7iXT71dcfmZmNgzNWahIeNJDD3
// SIG // 7zTQYx2xQmdKDku/Og7vtpU6pzjkJZIIpohmgjCCBbww
// SIG // ggOkoAMCAQICCmEzJhoAAAAAADEwDQYJKoZIhvcNAQEF
// SIG // BQAwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcGCgmS
// SIG // JomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // MB4XDTEwMDgzMTIyMTkzMloXDTIwMDgzMTIyMjkzMlow
// SIG // eTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWlj
// SIG // cm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCycllcGTBkvx2a
// SIG // YCAgQpl2U2w+G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/
// SIG // 3sJCTiPVcgDbNVcKicquIEn08GisTUuNpb15S3GbRwfa
// SIG // /SXfnXWIz6pzRH/XgdvzvfI2pMlcRdyvrT3gKGiXGqel
// SIG // cnNW8ReU5P01lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJp
// SIG // L9oZC/6SdCnidi9U3RQwWfjSjWL9y8lfRjFQuScT5EAw
// SIG // z3IpECgixzdOPaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn
// SIG // 0i1i8UU956wIAPZGoZ7RW4wmU+h6qkryRs83PDietHdc
// SIG // pReejcsRj1Y8wawJXwPTAgMBAAGjggFeMIIBWjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJ
// SIG // Ny4zFha5TJoKHzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGC
// SIG // NxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQU/dExTtMm
// SIG // ipXhmGA7qDFvpjy82C0wGQYJKwYBBAGCNxQCBAweCgBT
// SIG // AHUAYgBDAEEwHwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8
// SIG // KuEKU5VZ5KQwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvbWljcm9zb2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUF
// SIG // BwEBBEgwRjBEBggrBgEFBQcwAoY4aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRS
// SIG // b290Q2VydC5jcnQwDQYJKoZIhvcNAQEFBQADggIBAFk5
// SIG // Pn8mRq/rb0CxMrVq6w4vbqhJ9+tfde1MOy3XQ60L/svp
// SIG // LTGjI8x8UJiAIV2sPS9MuqKoVpzjcLu4tPh5tUly9z7q
// SIG // QX/K4QwXaculnCAt+gtQxFbNLeNK0rxw56gNogOlVuC4
// SIG // iktX8pVCnPHz7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y
// SIG // 4k74jKHK6BOlkU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wd
// SIG // zaKMvSeyeWNWRKJRzfnpo1hW3ZsCRUQvX/TartSCMm78
// SIG // pJUT5Otp56miLL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q7
// SIG // 0eFW6NB4lhhcyTUWX92THUmOLb6tNEQc7hAVGgBd3TVb
// SIG // Ic6YxwnuhQ6MT20OE049fClInHLR82zKwexwo1eSV32U
// SIG // jaAbSANa98+jZwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKN
// SIG // MxZlHg6K3RDeZPRvzkbU0xfpecQEtNP7LN8fip6sCvsT
// SIG // J0Ct5PnhqX9GuwdgR2VgQE6wQuxO7bN2edgKNAltHIAx
// SIG // H+IOVN3lofvlRxCtZJj/UBYufL8FIXrilUEnacOTj5XJ
// SIG // jdibIa4NXJzwoq6GaIMMai27dmsAHZat8hZ79haDJLmI
// SIG // z2qoRzEvmtzjcT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq
// SIG // /2mbluIQqBC0N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZo
// SIG // NAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYKCZIm
// SIG // iZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJbWlj
// SIG // cm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBD
// SIG // ZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1
// SIG // MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQrc7d
// SIG // GE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr
// SIG // 6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd6udO
// SIG // o6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOwhtDl
// SIG // KEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd+
// SIG // +NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi+YSn
// SIG // EUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68e
// SIG // eEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXDwiAL
// SIG // AgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/MB0G
// SIG // A1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzALBgNV
// SIG // HQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgGA1Ud
// SIG // IwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KShY6Rh
// SIG // MF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJ
// SIG // k/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eYIQ
// SIG // ea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BB
// SIG // hj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5jcmww
// SIG // VAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjhodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01p
// SIG // Y3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAKBggr
// SIG // BgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wD
// SIG // RDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQSooxt
// SIG // YrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ7DsB
// SIG // 7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mR
// SIG // KiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxnLcVR
// SIG // DupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+FChPf
// SIG // wgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4wyrt4
// SIG // ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGys
// SIG // OUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8csu89D
// SIG // s+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyzs7BN
// SIG // Z8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU
// SIG // 2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/m4LB
// SIG // J1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HIjCPJ
// SIG // Zr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCl
// SIG // eKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/J
// SIG // mu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0bodj6
// SIG // L54/LlUWa8kTo/0xggSDMIIEfwIBATCBkDB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQQITMwAAALARrwqL0Duf3QAB
// SIG // AAAAsDAJBgUrDgMCGgUAoIGcMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBSarml10pIe
// SIG // q+MSRg0lw12EZxDdNjA8BgorBgEEAYI3AgEMMS4wLKAS
// SIG // gBAAUgB1AGwAZQByAC4AagBzoRaAFGh0dHA6Ly9taWNy
// SIG // b3NvZnQuY29tMA0GCSqGSIb3DQEBAQUABIIBAOKZb0SS
// SIG // cB8mFTWIQOFktdKdx4hub1AdhlkuKr/jjUGvskjUUDib
// SIG // xxV4JkiOlpKyPRAtLB9IRGnkKPPomGnnfKcDdJMZiIoz
// SIG // ZHMD09Jadln14xhKUzeSBJHC0JO/N4LZ7hmnIrUNIFoi
// SIG // xASOpcNjILfrSceWSdxPweHgGaTip/67nClz/nM4Zq23
// SIG // l95n18Yq5I4QK3L9O6VLfOemSQEdTe7/T3u6jF1p2dTS
// SIG // NVdVUGyT8w9xRNy40jPE8GTz8YNwAjP4HQARpHBSWeLo
// SIG // EU0nZMDxb6b1xTerIoFPsrz0fxLm3F9YKI+R37B0RCbi
// SIG // u4eYoYhU3aqKgZdNMMpZDl2tm3qhggIoMIICJAYJKoZI
// SIG // hvcNAQkGMYICFTCCAhECAQEwgY4wdzELMAkGA1UEBhMC
// SIG // VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
// SIG // B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
// SIG // b3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRpbWUt
// SIG // U3RhbXAgUENBAhMzAAAAKzkySMGyyUjzAAAAAAArMAkG
// SIG // BSsOAwIaBQCgXTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcN
// SIG // AQcBMBwGCSqGSIb3DQEJBTEPFw0xMzAzMTUwNjM0MDJa
// SIG // MCMGCSqGSIb3DQEJBDEWBBQAaHSs6b0eXMeBwNr6m4hf
// SIG // VWgihTANBgkqhkiG9w0BAQUFAASCAQBtQy0d+cwdCmo1
// SIG // G4yGc07Oe/Tj7kIOp2UZ9MelC+ctD+reRJQkOPRP/w+2
// SIG // 9i/IvhH+eL72uLkF4WASu7cyn88blJGN3q/+jxODMRVa
// SIG // KOdg3jQ+qtqC4g5/zPIEXTFruh/1AxGe6eqQBO93VEQB
// SIG // 3lkrYfQFRkwmE8AwWC+ADl5kGcrTgaeq2m8DVLQ5sQyf
// SIG // SqfkDbPEP5J2+Lfa2jAs1Wz8c/ogDdyP/Hw9vBWs8HP3
// SIG // h/v7i366jgdInMJCUOZODyhAQnxgwi73JrePTrxaHVGT
// SIG // o5CmAF/ZGKUtWYwM1149jEom7XSP5+P1GzmXrfINPGXh
// SIG // TDI1taX8BUyyTYIX0+Dj
// SIG // End signature block
